# Copyright (c) 2018 SurePassID Corp
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are
# met:
#
#   * Redistributions of source code must retain the above copyright
#     notice, this list of conditions and the following disclaimer.
#
#   * Redistributions in binary form must reproduce the above
#     copyright notice, this list of conditions and the following
#     disclaimer in the documentation and/or other materials provided
#     with the distribution.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
# "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
# LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
# A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
# OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
# SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
# LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
# DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
# THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
# (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#######################################################################
# Module: rlm_surepassid.pl 
# Description: SurePassID MFA plug-in for freeradius 
# Version: 2019_01_03.1 
#####################################################################

use strict;
use warnings;
use vars qw(%RAD_REQUEST %RAD_REPLY %RAD_CHECK);

#use lib 'e:\unix\perl\rlm_surepassid';
#use Error qw(:try);

#Add script directory to @INC:
use File::Spec::Functions qw(rel2abs);
use File::Basename;
use lib dirname(rel2abs($0));

#print dirname(rel2abs($0)); 

use SurePassId;

########################
# FreeRADIUS constants #
########################

use constant    RLM_MODULE_REJECT=>    0;#  /* immediately reject the request */
use constant	RLM_MODULE_FAIL=>      1;#  /* module failed, don't reply */
use constant	RLM_MODULE_OK=>	2;#  /* the module is OK, continue */
use constant	RLM_MODULE_HANDLED=>   3;#  /* the module handled the request, so stop. */
use constant	RLM_MODULE_INVALID=>   4;#  /* the module considers the request invalid. */
use constant	RLM_MODULE_USERLOCK=>  5;#  /* reject the request (user is locked out) */
use constant	RLM_MODULE_NOTFOUND=>  6;#  /* user not found */
use constant	RLM_MODULE_NOOP=>      7;#  /* module succeeded without doing anything */
use constant	RLM_MODULE_UPDATED=>   8;#  /* OK (pairs modified) */
use constant	RLM_MODULE_NUMCODES=>  9;#  /* How many return codes there are */

# Radlog's constants
use constant L_DBG => 1;
use constant L_AUTH => 2;
use constant L_INFO => 3;
use constant L_ERR => 4;
use constant L_PROXY => 5;
use constant L_ACCT => 6;
use constant L_CONS => 128;

# supported token types 
use constant TOKENTYPE_None => -1;
use constant TOKENTYPE_Fob => 0;
use constant TOKENTYPE_Desktop => 1;
use constant TOKENTYPE_Blackberry => 2;
use constant TOKENTYPE_SmartPhone => 3;
use constant TOKENTYPE_ElectronicCard => 4;
use constant TOKENTYPE_SmartCard => 5;
use constant TOKENTYPE_MatrixCard => 6;
use constant TOKENTYPE_CellPhoneSms => 7;
use constant TOKENTYPE_Android => 8;
use constant TOKENTYPE_Ios => 9;
use constant TOKENTYPE_CellPhoneVoice => 10;
use constant TOKENTYPE_CellPhoneVoiceNoOtp => 11;
 
 # delivery supported otp types 
 
use constant OTPTYPE_Event => 0;
use constant OTPTYPE_Time => 1;
use constant OTPTYPE_TimeAndPasssword => 2;
use constant OTPTYPE_Ocra => 3;
use constant OTPTYPE_OcraAndPin => 4;
use constant OTPTYPE_OcraHiSense => 5;
use constant OTPTYPE_OcraHiSenseAndPin => 6;
use constant OTPTYPE_Matrix => 7; 

# delivery methods for send and push 

use constant SENDOTPDELIVERYMETHOD_None => -1;
use constant SENDOTPDELIVERYMETHOD_Sms => 0;
use constant SENDOTPDELIVERYMETHOD_Email => 1;
use constant SENDOTPDELIVERYMETHOD_Voice => 2;
use constant SENDOTPDELIVERYMETHOD_SmsPushQuestion => 3;
use constant SENDOTPDELIVERYMETHOD_PushFidoU2FQuestion => 4;
use constant SENDOTPDELIVERYMETHOD_PushAppQuestion => 5;
use constant SENDOTPDELIVERYMETHOD_PushOtp => 6;
use constant SENDOTPDELIVERYMETHOD_VoicePushQuestion => 7;

# SP return codes - short list 

use constant RESULTCODE_Success => 0;
use constant RESULTCODE_PartnerLoginFailed => 9000;
use constant RESULTCODE_PartnerUserLoginFailed => 9001;
use constant RESULTCODE_OTPAlreadyUsed => 9002;
use constant RESULTCODE_OTPInvalid => 9003;
use constant RESULTCODE_MaximumFailedOTPRequestsExceeded => 9015;
use constant RESULTCODE_PushExpiredFailure  => 9111;
use constant RESULTCODE_PushBadTooEarly => 9116;
use constant RESULTCODE_PinResetMode => 9122;
use constant RESULTCODE_Error8888 => 8888;
use constant RESULTCODE_Error9999 => 9999;
use constant RESULTCODE_Error9998 => 9998;
use constant RESULTCODE_Error9997 => 9997;
use constant RESULTCODE_Error9996 => 9996;
 

# rlm_surepassid reply messages  TODO -verify these messages are the correct text. 
use constant RADAPI_REPLY_NO_LOGIN_NAME => "Invalid username/password."; 
use constant RADAPI_REPLY_NO_LOGIN_PW => "Invalid username/password.";
use constant RADAPI_REPLY_OTP_NOT_ALLOWED => "This access request is not permitted. Please try a different request.";
use constant RADAPI_REPLY_NO_USER_FOR_CLIENT => "Invalid username/password"; 
use constant RADAPI_REPLY_NO_USER_OTP_DEVICE => "Invalid access request. No token for you."; 
use constant RADAPI_REPLY_SENDOTP_CHALLENGE_QUESTION_PROMPT => "Passcode/PIN has been sent."; 
use constant RADAPI_REPLY_SENDPUSH_CHALLENGE_QUESTION_PROMPT => "Access request  has been sent. Please approve and then resubmit."; 
use constant RADAPI_REPLY_ENTER_CHALLENGE_QUESTION_PROMPT => "Please enter your passcode/PIN."; 
use constant RADAPI_REPLY_MISSING_LOGIN_PW => "Password is missing or invalid.";
use constant RADAPI_REPLY_SENDPUSH_WAITING_APPROVAL => "Your access request has not been approved yet.";
use constant RADAPI_REPLY_SENDPUSH_DECLINED => "Your access request has been denied.";
use constant RADAPI_REPLY_SENDPUSH_FAILURE => "The access request  was invalid or failed.";
use constant RADAPI_REPLY_CHALLENGE_OTP_MISSING => "Please enter a passcode/PIN.";
use constant RADAPI_REPLY_CHALLENGE_OTP_FAILURE => "Invalid passcode/PIN. Please re-enter.";
use constant RADAPI_REPLY_SENDPUSH_NOT_ALLOWED => "This access request is not permitted. Please try a different request.";
use constant RADAPI_REPLY_SENDPUSH_TIMEOUT_FAILURE => "The access request expired. Please try again.";


# rlm_surepassid reply messages  TODO -verify these messages are the correct text. 
use constant DIRECTORY_TYPE_SurePass => "SurePassId"; 
use constant DIRECTORY_TYPE_AdLdap => "Ldap";
use constant DIRECTORY_TYPE_RadProxy => "RadProxy";
use constant DIRECTORY_TYPE_Ad => "Invalid Password";
use constant DIRECTORY_TYPE_None => "AD";



# Default configuration

my $user= ''; 
my $configfile = "radius.conf"; 
our $trace = 0;
our $directoryOption = 1;  
our $vml = "2019_01_03.1";
our $appName = "Remote Access";
our $authnReason = "Login";
our $relyingPartyUrl = "rp";
our $authServerUrl = "";
our $allowSMS=0;
our $allowEmail=0;
our $allowCall=0;
our $allowPushApp=0;
our $allowPushSMS=0;
our $allowPushOtp=0;
our $allowPushVoice=0;
our $allowPushAppU2F=0;
our $useUpn=0;
our $useGetVerifyMethods=0;
our $waitForPushResponse=1; 
our $tc=-1; 
our $OtpSplitChar = ",";

&radiusd::radlog(1, "RLM_SUREPASSID init() STARTING");

# read config parameters

SurePassID::WriteTrace($user, "main() - Stating version=$vml"); 
SurePassID::WriteTrace($user, "main() - Reading Config Parameters...."); 

my %params = SurePassID::ReadConfigParams($configfile, 1);
our %userpushstate; 
our %userstate; 

SurePassID::WriteTrace($user, "main() - Reading Config Parameters OK...."); 

# check config parameters

SurePassID::WriteTrace($user, "main() - Checking Config Parameters...."); 
my $paramsOk = SurePassID::CheckConfigParams(%params);

if ($paramsOk == 0)
{
    &radiusd::radlog(L_ERR, "main() - Checking Config Parameters()) FAILED. If TraceOn=1 then check SP trace file in /etc/surepassid/rlm. If not, set TraceOn=1 in /etc/surepassid/rlm/radius.conf, restart FreeRADIUS (radiusd) and review SP trace file. ");
    SurePassID::WriteTrace($user, "main() - Checking Config Parameters FAILED");
}

if (exists($params{'TraceOn'}))
{
   $trace = $params{'TraceOn'}; 
} 

&radiusd::radlog(L_INFO, "RLM_SUREPASSID main() - TraceOn=$trace");

if ($trace) 
{
   SurePassID::WriteTrace($user, "main() - Tracing is on");
} 

if (exists($params{'WaitForPushResponse'}))
{
   $waitForPushResponse = $params{'WaitForPushResponse'}; 
} 

SurePassID::WriteTrace($user, "main() - WaitForPushResponse - waitForPushResponse=$waitForPushResponse");
SurePassID::WriteTrace($user, "main() - Getting Optional Config Parameters");

get_options(); 

$authServerUrl = adjust_server_endpoint($params{'AuthServerURL'}); 
SurePassID::WriteTrace($user, "main() - Server EndPoint: $authServerUrl");

if ($relyingPartyUrl eq "") 
{
   $relyingPartyUrl = $authServerUrl; 
} 

SurePassID::WriteTrace($user, "main() - Waiting for requests...");

sub WriteTrace($$) 
{
  if ($trace==1) 
  {
    SurePassID::WriteTrace( $_[0],  $_[1]);
  }
} 

sub get_options() 
{
# get security settings

if (exists($params{'AllowSMS'}))
{
   $allowSMS = $params{'AllowSMS'}; 
} 
if (exists($params{'AllowEmail'}))
{
   $allowEmail = $params{'AllowEmail'}; 
} 
if (exists($params{'AllowCall'}))
{
   $allowCall = $params{'AllowCall'}; 
} 
if (exists($params{'AllowPushApp'}))
{
   $allowPushApp = $params{'AllowPushApp'}; 
} 
if (exists($params{'AllowPushSMS'}))
{
   $allowPushSMS = $params{'AllowPushSMS'}; 
} 
if (exists($params{'AllowPushOtp'}))
{
   $allowPushOtp = $params{'AllowPushOtp'}; 
} 
if (exists($params{'AllowPushVoice'}))
{
   $allowPushVoice= $params{'AllowPushVoice'}; 
} 
if (exists($params{'AllowPushAppU2F'}))
{
   $allowPushAppU2F= $params{'AllowPushAppU2F'}; 
} 
if (exists($params{'SurePassUseUpn'})) 
{
   $useUpn= $params{'SurePassUseUpn'}; 
} 
if (exists($params{'SurePassUseVerifyMethod'})) 
{
   $useGetVerifyMethods= $params{'SurePassUseVerifyMethod'}; 
} 
if (exists($params{'PWOtpSplitChar'})) 
{
   $OtpSplitChar = $params{'PWOtpSplitChar'}; 
} 



SurePassID::WriteTrace($user, "get_options() - Send OTP Security: AllowSMS=$allowSMS AllowEmail=$allowEmail AllowCall=$allowCall ");
SurePassID::WriteTrace($user, "get_options() - Send Push Security: AllowPushApp=$allowPushApp AllowPushSMS=$allowPushSMS AllowPushOtp=$allowPushOtp AllowPushVoice=$allowPushVoice AllowPushAppU2F=$allowPushAppU2F");
SurePassID::WriteTrace($user, "get_options() - Additional Options: SurePassUseVerifyMethod=$useGetVerifyMethods SurePassUseUpn=$useUpn OtpSplitChar=$OtpSplitChar");

# get push params 

if (exists($params{'PushAppName'}))
{
   $appName = $params{'PushAppName'}; 
} 
if (exists($params{'PushAuthnReason'}))
{
   $authnReason  = $params{'PushAuthnReason'}; 
} 
if (exists($params{'RelyingPartyUrl'}))
{
   $relyingPartyUrl = $params{'RelyingPartyUrl'}; 
} 

SurePassID::WriteTrace($user, "get_options() - Push Options: RelyingPartyUrl=$relyingPartyUrl PushAuthnReason=$authnReason PushAppName=$appName ");

}
sub is_send_push_request_allowed($) 
{
   my $request = $_[0];

   if ($request == SENDOTPDELIVERYMETHOD_Sms )
   {
     return $allowSMS; 
   } 
  if ($request == SENDOTPDELIVERYMETHOD_Email)
   {
     return $allowEmail; 
   } 
  if ($request == SENDOTPDELIVERYMETHOD_Voice)
   {
     return $allowCall; 
   } 
  if ($request == SENDOTPDELIVERYMETHOD_SmsPushQuestion)
   {
     return $allowPushSMS; 
   } 
  if ($request == SENDOTPDELIVERYMETHOD_PushFidoU2FQuestion)
   {
     return $allowPushAppU2F; 
   } 
  if ($request == SENDOTPDELIVERYMETHOD_PushAppQuestion)
   {
     return $allowPushApp 
   } 
  if ($request == SENDOTPDELIVERYMETHOD_PushOtp)
   {
     return $allowPushOtp; 
   } 
  if ($request == SENDOTPDELIVERYMETHOD_VoicePushQuestion)
   {
     return $allowPushVoice; 
   } 

   return 1; 

} 
sub adjust_server_endpoint($) 
{
   my $endpoint = $_[0]; 

   if ($endpoint eq "sandbox") 
   {
      $endpoint =  "https://sandbox.surepassid.com/AuthServer/REST/OATH/OATHServer.aspx";
   } 
   elsif ($endpoint eq "prod" || $endpoint eq "cloud" || $endpoint eq "production") 
   {
      $endpoint =  "https://cloud.surepassid.com/AuthServer/REST/OATH/OATHServer.aspx";
   } 

   return ($endpoint); 
} 
sub get_user_state_teable($) 
{
       my $username = $_[0];
       my $val = ""; 
       if (exists($userpushstate{$username}))
        {
          $val = $userpushstate{$username};
        }  

   return ($val); 
} 
sub get_state_table($) 
{
       my $username = $_[0];
       my $val = "0"; 
       if (exists($userstate{$username}))
        {
          $val = $userstate{$username};
        }  
        

        return ($val); 
} 
sub delete_user_state_table($) 
{
   my $username = $_[0];

   if (exists($userpushstate{$username}))
   {
     delete($userpushstate{$username}); 
   } 

} 

sub update_user_state_table($$) 
{
       my $username = $_[0];
       my $val =  $_[1];

       $userpushstate{$username} = $val;
       WriteTrace($user, "userpushstate{$username} = $val "); 

       return ($val); 
}  
sub update_state_table($$) 
{
       my $username = $_[0];
       my $val =  $_[1];

       $userstate{$username} = $val;
       return ($val); 
}  

#
#  alternate implentation 
#

sub get_send_otp_pushMethod($) 
{

   my $rq = $_[0]; 
     
   if ($rq eq "!") 
	{
		return SENDOTPDELIVERYMETHOD_PushAppQuestion;
	}
        elsif ($rq eq "#")
	{
		return SENDOTPDELIVERYMETHOD_VoicePushQuestion;
	}

	
	return SENDOTPDELIVERYMETHOD_None; 
} 

#
#  standard implentation 
#

sub get_send_otp_pushMethod_original($) 
{

   my $rq = $_[0]; 
   my $rqu = uc($_[0]);
   
    if ($rqu eq "SENDSMS" || $rq eq "#" || $rq eq "0")
	{
		return SENDOTPDELIVERYMETHOD_Sms;  
	}
	elsif ($rqu eq "SENDEMAIL" || $rqu eq "M" || $rq eq "1")
	{
		return SENDOTPDELIVERYMETHOD_Email; 
	}
	elsif ($rqu eq "SENDVOICE" || $rqu eq "V" || $rq eq "2")
	{
		return SENDOTPDELIVERYMETHOD_Voice;
	}
	elsif ($rqu eq "PUSHAPPFIDOU2F" || $rqu eq "U" || $rq eq "4")
	{
		return SENDOTPDELIVERYMETHOD_PushFidoU2FQuestion;
	}
	elsif ($rqu eq "PUSHAPPQUESTION" || $rq eq "?" || $rqu eq "Q" || $rq eq "5" || $rqu eq "PUSHAPP")
	{
		return SENDOTPDELIVERYMETHOD_PushAppQuestion;
	}
	elsif ($rqu eq "PUSHOTP" || $rqu eq "P" || $rq eq "6")
	{
		return SENDOTPDELIVERYMETHOD_PushOtp;
	}
 	elsif ($rqu eq "PUSHVOICE" || $rqu eq "#" || $rq eq "7")
	{
		return SENDOTPDELIVERYMETHOD_VoicePushQuestion;
	}

	
	return SENDOTPDELIVERYMETHOD_None; 
} 
sub split_domain ($)
{

    #split fqdn into subsequent pieces
		
	my $fqdn = $_[0]; 
	my $domain  = "."; 
	my $username = ""; 
	my @parts; 
	
	$fqdn =~ tr#\\#@#;
	@parts = split(/@/, $fqdn, 2);
	
	 if (@parts == 2) 
	 {
		$domain = $parts[1]; 
		$username = $parts[0]; 
		return ($domain, $username); 
	 }
	 
	 return ($domain, $fqdn); 
		
} 

sub authenticate {
	
	my $sessionToken = "";
	my $authnReqId = "";
        my $otp_len = 6; 
  	my $errorMsg = ""; 
        my $errorCode = 0; 
	my $domain = ""; 
	my $ioption = 0; 
	my $username = "";
	my $state = "0";  	
	my $userauthed = 0;  	
	my $password = '';   
        my $defaultAuthCommand = SENDOTPDELIVERYMETHOD_None; 
        my $isOtpAllowed= 1; 
	my $methods; 
        
        my $otp =""; 
        my $splitpassword =""; 

	
	if ($paramsOk == 0)
	{
	    WriteTrace($user, "authenticate() - Checking Config Parameters() FAILED"); 
	    &radiusd::radlog(L_ERR, "Checking Config Parameters()) FAILED. Check SP trace file");
	    $RAD_REPLY{'Reply-Message'} = "Checking Config Parameters() FAILED";
	    return RLM_MODULE_REJECT;
	}
    
	my $fqdn = $RAD_REQUEST{'User-Name'};
	WriteTrace($user, "authenticate() - ********** Start processing new request fqdn=$fqdn **********"); 
	
	if (length($RAD_REQUEST{'User-Name'}) == 0) 
	{
	    $RAD_REPLY{'Reply-Message'} = RADAPI_REPLY_NO_LOGIN_NAME;
	    return RLM_MODULE_REJECT;
	}
	
        #
 	# split AD style username and password 
        #

	if ($useUpn)
	{
	   $username = $fqdn;
	}
	else 
	{
		($domain, $username) = split_domain($fqdn); 
	} 

	WriteTrace($user, "authenticate() -  = $domain username=$username"); 
	
        $state = get_state_table($username); 
  	WriteTrace($user, "authenticate() - userstate{'$username'} = $state "); 
 
	#$password = $RAD_REQUEST{'User-Password'}; 
	
        #
        # see if we have a split characer in the password. 
        #

        ($password , $otp) = SplitPassword($RAD_REQUEST{'User-Password'}, $OtpSplitChar);

	($errorCode, $errorMsg, $sessionToken) = SurePassID::ValidateUser($authServerUrl, $params{'AuthServerToken'}, $params{'AuthServerKey'}, $username, $password, $trace);
	if ($errorCode != RESULTCODE_Success)
	{
	    $RAD_REPLY{'Reply-Message'} = RADAPI_REPLY_NO_LOGIN_PW;
	    return RLM_MODULE_REJECT;
	} 

        ##############################################################################################
        #  START - See if there are any verify methods available and any valid manual input requests 
        ##############################################################################################

           my $sendpushoption; 

           if ($useGetVerifyMethods == 0) 
    	   {
               $sendpushoption = get_send_otp_pushMethod_original($otp); 
               if ($sendpushoption == SENDOTPDELIVERYMETHOD_None)
               {
                   $sendpushoption = get_send_otp_pushMethod_original($password); 
               }
               else
               {
                   $otp = ""; 
               }
                
	       WriteTrace($user, "authenticate() - useGetVerifyMethods = 0 get_send_otp_pushMethodoriginal()=$sendpushoption"); 

               if (is_send_push_request_allowed($sendpushoption) == 0) 
	       {
		  WriteTrace($user, "authenticate() - Send/Push Option $sendpushoption is not enabled!"); 
		  $RAD_REPLY{'Reply-Message'} = RADAPI_REPLY_SENDPUSH_NOT_ALLOWED;
		  return RLM_MODULE_REJECT;
	       }

               $defaultAuthCommand = $sendpushoption; 
	     
           }
           else 
   	   {
	       $sendpushoption = get_send_otp_pushMethod($otp); 
	       WriteTrace($user, "authenticate() - useGetVerifyMethods = 1 get_send_otp_pushMethod()=$sendpushoption"); 
            
	       WriteTrace($user, "authenticate() - Processing VerifyMethod() - Get VerifyMethods from the server"); 
	       ($errorCode, $errorMsg, $methods) = SurePassID::VerifyMethod($params{'AuthServerURL'}, $params{'AuthServerToken'}, $params{'AuthServerKey'}, $username, $trace);
	       WriteTrace($user, "SurePassID::VerifyMethod response is $errorCode message=$errorMsg"); 
             
               if ($tc < 10) 
               {
 	              ($defaultAuthCommand, $isOtpAllowed) = RunTestCase($tc); 
	    	      SurePassID::WriteTrace($user, "authenticate() - GetVerifyMethods TESTCASE OVERRIDE tc=$tc defaultAuthCommand = [$defaultAuthCommand] isOtpAllowed=[$isOtpAllowed]"); 
	       }
               else
               {
 	              $methods = GetMethodsTestData($tc); 
	    	      SurePassID::WriteTrace($user, "authenticate() - GetVerifyMethods TESTCASE OVERRIDE tc=$tc"); 
	       }
 

	       if ($errorCode == RESULTCODE_Success) 
	        {   
	    	   WriteTrace($user, "authenticate() - calling ParseVerifyMethod()"); 
	           ($defaultAuthCommand, $isOtpAllowed) = ParseVerifyMethod($methods);
	    	   WriteTrace($user, "authenticate() - ParseVerifyMethod forced defaultAuthCommand = [$defaultAuthCommand] isOtpAllowed=[$isOtpAllowed]"); 
  
                   if ($sendpushoption != SENDOTPDELIVERYMETHOD_None)
                   {
                      $otp = ""; 
		      WriteTrace($user, "authenticate() - GetVerifyMethods defaultAuthCommand override = [$defaultAuthCommand] with =[$sendpushoption]"); 
                      if (IsMethodAllowed($methods, $sendpushoption) == 0) 
                      {
		         WriteTrace($user, "authenticate() - IsMethodAllowed($sendpushoption)==0"); 
		         $RAD_REPLY{'Reply-Message'} = RADAPI_REPLY_SENDPUSH_NOT_ALLOWED . $otp;
		         return RLM_MODULE_REJECT;
                       }

                       $defaultAuthCommand = $sendpushoption; 
                   } 
                   else 
                   {
                       # if we have an otp call off automatic methods 
		      WriteTrace($user, "authenticate() - otp = [$otp] "); 

                       if ($otp ne "") 
                       {
 		          WriteTrace($user, "authenticate() - setting default command SENDOTPDELIVERYMETHOD_None"); 
                         $defaultAuthCommand = SENDOTPDELIVERYMETHOD_None;
                       } 
                   } 
                } 
            }
 

        ##############################################################################################
        #  START - Process send otp 
        ##############################################################################################
    	
	  if ($sendpushoption == SENDOTPDELIVERYMETHOD_Sms || 
	      $sendpushoption == SENDOTPDELIVERYMETHOD_Email || 
	      $sendpushoption == SENDOTPDELIVERYMETHOD_Voice) 
	   {
		    WriteTrace($user, "authenticate() - SendOtp deliveryoption = $sendpushoption "); 
		    delete_user_state_table($username); 

		    ($errorCode, $errorMsg, $sessionToken) = SurePassID::SendOtp($authServerUrl, $params{'AuthServerToken'}, $params{'AuthServerKey'}, $username, $sendpushoption, "", $trace);
		    if ($errorCode != RESULTCODE_Success)
		    {
			  WriteTrace($user, "authenticate() - SendOtp to $username FAILED = $errorMsg"); 
			  $RAD_REPLY{'Reply-Message'} = $errorMsg;
			  return RLM_MODULE_REJECT;
		    }
	
		    WriteTrace($user, "authenticate() - SendOtp to $username OK"); 

		   $RAD_REPLY{'Reply-Message'} = RADAPI_REPLY_SENDOTP_CHALLENGE_QUESTION_PROMPT; 
		   $RAD_CHECK{'Response-Packet-Type'} = "Access-Challenge"; 
		   return RLM_MODULE_HANDLED;
		
	    } 
		
        ##############################################################################################
        #  END - Process user manually requested a push and there is no verify_methods override 
        ##############################################################################################

        ##############################################################################################
        #  START - CHallenge Response part two - OTP 
        ##############################################################################################

	#just check OTP since username and password were validated and the user was sent a challenge 
	if ($state eq "1" ) 
	{
		WriteTrace($user, "authenticate() - state==1 Process OTP challenge"); 

		if (length($RAD_REQUEST{'User-Password'}) == 0 && $authnReqId eq "") 
		{
		   WriteTrace($user, "authenticate() - Process Challenge for $username - Blank OTP. ERROR"); 
		   $RAD_REPLY{'Reply-Message'} = RADAPI_REPLY_CHALLENGE_OTP_MISSING;
		   return RLM_MODULE_REJECT;
		 }

		 ($errorCode, $errorMsg) = SurePassID::ValidateOtp($authServerUrl, $params{'AuthServerToken'}, $params{'AuthServerKey'}, $username,  "", $RAD_REQUEST{'User-Password'},$trace);
		 if ($errorCode == RESULTCODE_Success)
		 {
		   WriteTrace($user, "authenticate() - Process Challenge for $username - OTP OK"); 
		   $RAD_REPLY{'Reply-Message'} = "OK";
		   delete_user_state_table($username);
		   return RLM_MODULE_OK;
		 } 
		 else 
		 {
		   WriteTrace($user, "authenticate() - Process Challenge for $username. OTP Invalid"); 
		   $RAD_REPLY{'Reply-Message'} = RADAPI_REPLY_CHALLENGE_OTP_FAILURE;
		   return RLM_MODULE_REJECT;
		 } 
	} 

        ##############################################################################################
        #  END - Challenge Response part two - OTP 
        ##############################################################################################


	update_state_table($username, "0");
	$authnReqId = get_user_state_teable($username);
	WriteTrace($user, "authnReqId = $authnReqId"); 

	if (length($RAD_REQUEST{'User-Password'}) == 0 && $authnReqId eq "") 
	{
		WriteTrace($user, "authenticate() - No password and no Push in progress for $username. ERROR"); 
		$RAD_REPLY{'Reply-Message'} = RADAPI_REPLY_NO_LOGIN_PW;
		return RLM_MODULE_REJECT;
	}

        ##############################################################################################
        #  START - Process implied push mechanism 
        ##############################################################################################
  
       #
       # if otp is blank then lets' try the whole password.  if otp is not blank then theh bypass implied push 
       # 
         
        if ($defaultAuthCommand != SENDOTPDELIVERYMETHOD_None)       
        {
 
  		 
			 ($errorCode, $errorMsg, $authnReqId) = ProcessVerifyMethods($defaultAuthCommand, $username);
			  if ($errorCode != RESULTCODE_Success)
			  {
				WriteTrace($user, "authenticate() - SendPush to $username FAILED = $errorMsg. ERROR"); 
				$RAD_REPLY{'Reply-Message'} = $errorMsg;
				return RLM_MODULE_REJECT;
			  }
			  else 
			  {
			         WriteTrace($user, "authenticate() - SendPush to $username OK = $errorMsg"); 
				 if ($waitForPushResponse == 1) 
				 {
				 	WriteTrace($user, "authenticate() - Calling ValidatePushResponse() waitForPushResponse=" . $waitForPushResponse); 
                                        if ($defaultAuthCommand == SENDOTPDELIVERYMETHOD_PushOtp) 
                                        {
		                           return ValidatePushResponse($username, $authnReqId, 0);
                                        }
                                        else
                                        {
		                           return ValidatePushResponse($username, $authnReqId, $waitForPushResponse);
                                        }
 		                 }
				 
				 WriteTrace($user, "authenticate() - Sending challenge" . $waitForPushResponse); 
		 	         update_user_state_table($username, $authnReqId); 
		                 $RAD_REPLY{'Reply-Message'} = RADAPI_REPLY_SENDPUSH_CHALLENGE_QUESTION_PROMPT; 
                                 $RAD_CHECK{'Response-Packet-Type'} = "Access-Challenge"; 
				 return RLM_MODULE_HANDLED;
			  } 

        } 

        ##############################################################################################
        #  END - Process implied push mechanism 
        ##############################################################################################

        ##############################################################################################
        #  START - Wait for push response 
        ##############################################################################################

	# username + password does not match. implies OTP is added to password field or second part of standard push
	    
       if ($authnReqId ne "")
       { 
	 WriteTrace($user, "authenticate() - Process push OTP validate"); 

	  if ($userauthed) 
	  {
	  	 WriteTrace($user, "authenticate() - calling ValidatePushResponse() waitForPushResponse=0"); 
		 return ValidatePushResponse($username, $authnReqId, 0);
	  }
	  else 
	  {
		  delete_user_state_table($username);
		  WriteTrace($user, "authenticate() - Process push OTP validate. ERROR=Invalid username & password"); 
		  $RAD_REPLY{'Reply-Message'} = RADAPI_REPLY_NO_LOGIN_PW;
		  return RLM_MODULE_REJECT;
	  } 
        } 


        ##############################################################################################
        #  END - Wait for push response 
        ##############################################################################################

        ##############################################################################################
        #  START - Process OTP 
        ##############################################################################################

         WriteTrace($user, "authenticate() - Process PW+OTP"); 
                
        #
        #  validate user 
        #

         #if ($otp eq "") 
         #{
	 #   WriteTrace($user, "authenticate() - No OTP from split via char. reverting to standard password/otp split"); 
 
         #   my $password_len = length($RAD_REQUEST{'User-Password'}) - $otp_len;
         #   if ($password_len < 0) 
	 #   {
	 #      $RAD_REPLY{'Reply-Message'} = RADAPI_REPLY_NO_LOGIN_PW;
	 #      return RLM_MODULE_REJECT;
	 #   }
           
         #   $otp = substr $RAD_REQUEST{'User-Password'}, $password_len;
	 #   $password = substr $RAD_REQUEST{'User-Password'}, 0, $password_len;
         #}


         if (length($password) == length($RAD_REQUEST{'User-Password'}))
         {

                 WriteTrace($user, "authenticate() - User autheitcated and password match. Sending challenge" . $waitForPushResponse); 
		 update_user_state_table($username, $authnReqId); 
		 $RAD_REPLY{'Reply-Message'} = RADAPI_REPLY_CHALLENGE_OTP_MISSING; 
                 $RAD_CHECK{'Response-Packet-Type'} = "Access-Challenge"; 
		 return RLM_MODULE_HANDLED;
          }

         #
         # if otp is not allowed give error and return 
         #

         if ($isOtpAllowed==0) 
	 {
	   WriteTrace($user, "Process PW+OTP - OTP is not permitted for authentication"); 
	   $RAD_REPLY{'Reply-Message'} = RADAPI_REPLY_OTP_NOT_ALLOWED;
	   return RLM_MODULE_REJECT;
	 }

         #
         # if there is not otp from the split then split it here assuming the otp is 6 characters
         #
        
	 if (length($otp) < $otp_len) 
	 {
	    $RAD_REPLY{'Reply-Message'} = RADAPI_REPLY_NO_LOGIN_PW;
	    return RLM_MODULE_REJECT;
	 }

        #
        #  validate otp  
        #

        ($errorCode, $errorMsg) = SurePassID::ValidateOtp($authServerUrl, $params{'AuthServerToken'}, $params{'AuthServerKey'}, $username, "", $otp, $trace);
	if ($errorCode != RESULTCODE_Success)
	{
               if ($errorCode == RESULTCODE_OTPAlreadyUsed|| $errorCode == RESULTCODE_OTPInvalid|| $errorCode == RESULTCODE_MaximumFailedOTPRequestsExceeded)
                {
                    $RAD_REPLY{'Reply-Message'} = "Passcode is invalid. Please verify the code you entered is correct";
                }
                else 
                {
                    $RAD_REPLY{'Reply-Message'} = "Passcode is invalid. Error=$errorCode ";
               } 
            
	    return RLM_MODULE_REJECT;
	} 
    
        WriteTrace($user, "authenticate() - Process PW+OTP IS OK"); 

	$RAD_REPLY{'Reply-Message'} = "OK";
	return RLM_MODULE_OK;
	 
}
sub GetMethodsTestData($)
{
    my $tc = $_[0];
    my @methods = []; 

     WriteTrace($user, "GetMethodsTestData() ENTRY tc=$tc"); 

    if ($tc == 10) 
    {
     $methods[0] = 'PUSHAPP'; 
    } 
    elsif ($tc == 11 ) 
    {
       $methods[0] = 'PUSHAPP'; 
       $methods[1] = 'OTP'; 
    } 
    elsif ($tc == 12 ) 
    {
       $methods[0] = 'PUSHVOICE'; 
   } 
   elsif ($tc == 13 ) 
  {
  $methods[0] = 'PUSHVOICE'; 
  $methods[1] = 'OTP'; 
  } 
    elsif ($tc == 14 ) 
  {
   $methods[0] = 'OTP'; 
  } 
  elsif ($tc == 15 ) 
  {
  $methods[0] = 'x'; 
  } 
  elsif ($tc == 16 ) 
  {
  $methods[0] = 'PUSHVOICE'; 
  $methods[1] = 'PUSHAPP'; 
  $methods[2] = 'OTP'; 
  }  
   elsif ($tc == 17 ) 
  {
  $methods[0] = 'PUSHAPP'; 
  $methods[1] = 'PUSHVOICE'; 
  $methods[2] = 'OTP'; 
  } 
 elsif ($tc == 18) 
  {
  $methods[0] = 'PUSHVOICE'; 
  $methods[1] = 'PUSHAPP'; 
  } 
 elsif ($tc == 19) 
  {
  $methods[0] = 'PUSHAPP'; 
  $methods[1] = 'PUSHVOICE'; 

  } 


     WriteTrace($user, "GetMethodsTestData() EXIT"); 
    return (\@methods); 
}
sub RunTestCase($)
{
  my $tc = $_[0]; 
  my $defaultAuthCommand = SENDOTPDELIVERYMETHOD_None; 
  my $isOtpAllowed= 1; 
 
  if ($tc == 0 ) 
  {
     $defaultAuthCommand = SENDOTPDELIVERYMETHOD_VoicePushQuestion; 
     $isOtpAllowed= 1; 
  } 
  elsif ($tc == 1 ) 
  {
   $defaultAuthCommand = SENDOTPDELIVERYMETHOD_PushAppQuestion; 
   $isOtpAllowed= 1; 
  } 
   elsif ($tc == 2 ) 
  {
   $defaultAuthCommand = SENDOTPDELIVERYMETHOD_VoicePushQuestion; 
   $isOtpAllowed= 0; 
  } 
   elsif ($tc == 3 ) 
  {
   $defaultAuthCommand = SENDOTPDELIVERYMETHOD_PushAppQuestion; 
   $isOtpAllowed= 0; 
  } 
    elsif ($tc == 4 ) 
  {
    $isOtpAllowed= 1; 
  } 
  elsif ($tc == 5 ) 
  {
   $isOtpAllowed= 0; 
  } 
  else 
  {
    WriteTrace($user, "RunTestCase() - Testcase $tc is invalid and ignored"); 
    return ($defaultAuthCommand, $isOtpAllowed); 
  } 
  
  WriteTrace($user, "RunTestCase() - Testcase $tc override - defaultAuthCommand =$defaultAuthCommand, isOtpAllowed=$isOtpAllowed"); 
  return ($defaultAuthCommand, $isOtpAllowed); 

} 
sub ValidatePushResponse($$$)
{
	my $username = $_[0]; 
        my $authnReqId = $_[1];
	my $wait    = $_[2];
        my $errorMsg = ""; 
        my $errorCode = 0; 
	my $rlm_module_code = 0; 
	my $firstTime = 1; 
	
	while($wait || $firstTime) 
	{
	  $firstTime = 0; 
	   
	  ($errorCode, $errorMsg) = SurePassID::ValidateOtp($authServerUrl, $params{'AuthServerToken'}, $params{'AuthServerKey'}, $username, "", $authnReqId,  $trace);
	  if ($errorCode == RESULTCODE_Success)
		 {
			 WriteTrace($user, "ValidatePushResponse() - Process $authnReqId OK"); 
			 update_state_table($username, "0");
			 $RAD_REPLY{'Reply-Message'} = "OK";
			 $rlm_module_code = RLM_MODULE_OK;
			 last; 

		 } 
		 elsif ($errorCode == RESULTCODE_PushExpiredFailure )
		 {
			WriteTrace($user, "ValidatePushResponse() - Validate Push Token ERROR=Expired."); 
			delete_user_state_table($username);
			$RAD_REPLY{'Reply-Message'} = RADAPI_REPLY_SENDPUSH_TIMEOUT_FAILURE;
			$rlm_module_code =  RLM_MODULE_REJECT;
			last; 
		 }

		 elsif ($errorCode == RESULTCODE_PushBadTooEarly)
		 {
		   WriteTrace($user, "ValidatePushResponse() - Validate Push Token. ERROR=Waiting on user approval."); 
		   $RAD_REPLY{'Reply-Message'} = RADAPI_REPLY_SENDPUSH_WAITING_APPROVAL;
		   if ($wait == 0) 
		   {
			 $rlm_module_code =  RLM_MODULE_HANDLED;
                         last; 
		   } 

		 }
                 elsif ($errorCode == RESULTCODE_OTPInvalid || $errorCode == RESULTCODE_MaximumFailedOTPRequestsExceeded)
		 {
		   WriteTrace($user, "ValidatePushResponse() - Validate Push Token. ERROR=Declined"); 
		   $RAD_REPLY{'Reply-Message'} = RADAPI_REPLY_SENDPUSH_DECLINED;
                   $rlm_module_code =  RLM_MODULE_REJECT;
                   last; 
		 }

		 else 
		 {
			delete_user_state_table($username);
			WriteTrace($user, "Validate Push OTP ERROR. "); 
			$RAD_REPLY{'Reply-Message'} = $RAD_REPLY{'Reply-Message'} = RADAPI_REPLY_SENDPUSH_FAILURE;
			$rlm_module_code =  RLM_MODULE_REJECT;
			last; 
		 } 
		 
		 WriteTrace($user, "ValidatePushResponse() sleeping. snore snore."); 
		 sleep(1); 
	} 
	
	WriteTrace($user, "ValidatePushResponse() returnCode=" . $rlm_module_code); 
	return ($rlm_module_code); 
} 
sub ProcessVerifyMethods($$)
{
    my $pushoption = $_[0]; 
    my $username = $_[1]; 
    my $authnReqId = "";
    my $errorMsg = ""; 
    my $errorCode = 0; 
	


    if ($pushoption != SENDOTPDELIVERYMETHOD_PushAppQuestion &&
        $pushoption != SENDOTPDELIVERYMETHOD_VoicePushQuestion &&
        $pushoption != SENDOTPDELIVERYMETHOD_PushFidoU2FQuestion &&        
        $pushoption != SENDOTPDELIVERYMETHOD_SmsPushQuestion &&     
        $pushoption != SENDOTPDELIVERYMETHOD_PushOtp) 
	{
	    WriteTrace($user, "ProcessVerifyMethods() - to $username FAILED = Invalid push method=$pushoption. ERROR"); 
	    return ($authnReqId); 
	} 
	
        $pushoption = $pushoption -3; 
	
	($errorCode, $errorMsg, $authnReqId) = SurePassID::SendPush($authServerUrl, $params{'AuthServerToken'}, $params{'AuthServerKey'}, $username, $pushoption, $appName, $authnReason, $username, $relyingPartyUrl, $trace);
	if ($errorCode != RESULTCODE_Success)
	{
	    WriteTrace($user, "ProcessVerifyMethods() - SendPush to $username FAILED = $errorMsg. ERROR"); 
	}
	else 
	{
	    WriteTrace($user, "ProcessVerifyMethods() - ProcessVerifyMethods to $username OK = authnReqId=$authnReqId"); 
	}

	return ($errorCode, $errorMsg, $authnReqId); 
} 
sub SplitPassword($$)
 {
	
	my $pwotp = $_[0]; 
	my $splitchar = $_[1]; 
	my $pw = ""; 
	my $otp = ""; 
	        
	my $index = rindex($pwotp, $splitchar);
	if($index == -1)
	{
	    return ($pwotp,$otp);
	} 

        if ($index == 0) 
	{
	   return ("", substr($pwotp, 1, length($pwotp)-1));
	} 

    $pw = substr($pwotp, 0, $index);
	
	 return ($pw, substr($pwotp, $index+1, length($pwotp)-length($pw)-1));
   }

 sub IsMethodAllowed($$) 
 {
 
    my $methods = $_[0]; 
    my $requestedMethods = $_[1]; 
    my $index = 0; 
    my $isAllowed= 0; 
  
    my $size = @$methods;
  
        WriteTrace($user, "IsMethodAllowed() methods=$size requestedMethods=$requestedMethods"); 

	for($index=0; $index<$size; $index++)
	{  	
		WriteTrace($user, "IsMethodAllowed() - method=@$methods[$index] requestedMethods=$requestedMethods" ); 

		if (@$methods[$index] eq 'PUSHVOICE' && $requestedMethods == SENDOTPDELIVERYMETHOD_VoicePushQuestion) 
		{
			$isAllowed= 1; 
                        WriteTrace($user, "IsMethodAllowed() PUSHVOICE is allowed"); 
                        last;
		} 
                elsif (@$methods[$index] eq 'PUSHAPP' && $requestedMethods == SENDOTPDELIVERYMETHOD_PushAppQuestion) 
		{
			$isAllowed= 1; 
                        WriteTrace($user, "IsMethodAllowed() PUSHAPP is allowed"); 
                        last;
		} 
               elsif (@$methods[$index] eq 'PUSHOTP' && $requestedMethods == SENDOTPDELIVERYMETHOD_PushOtp ) 
		{
			$isAllowed= 1; 
                        WriteTrace($user, "IsMethodAllowed() PUSHAPP is allowed"); 
                        last;
		} 
                elsif (@$methods[$index] eq 'PUSHAPPFIDOU2F' && $requestedMethods == SENDOTPDELIVERYMETHOD_PushFidoU2FQuestion) 
		{
			$isAllowed= 1; 
                        WriteTrace($user, "IsMethodAllowed() PUSHAPP is allowed"); 
                        last;
		} 
              elsif (@$methods[$index] eq 'SENDVOICE' && $requestedMethods == SENDOTPDELIVERYMETHOD_Voice) 
		{
			$isAllowed= 1; 
                        WriteTrace($user, "IsMethodAllowed() SENDVOICE is allowed"); 
                        last;
		} 

              elsif (@$methods[$index] eq 'SENDEMAIL' && $requestedMethods == SENDOTPDELIVERYMETHOD_Email) 
		{
			$isAllowed= 1; 
                        WriteTrace($user, "IsMethodAllowed() SENDEMAIL is allowed"); 
                        last;
		} 
                elsif (@$methods[$index] eq 'SENDSMS' && $requestedMethods == SENDOTPDELIVERYMETHOD_Sms) 
		{
			$isAllowed= 1; 
                        WriteTrace($user, "IsMethodAllowed() SENDSMS is allowed"); 
                        last;
		} 
                else 
		{
                        WriteTrace($user, "IsMethodAllowed() @$methods[$index] is NOT allowed"); 
		} 


	} 
  
	
	

	return ($isAllowed) 
  } 

sub ParseVerifyMethod($) 
 {
 
    my $methods = $_[0]; 
    my $index = 0; 
    my $defaultAuthCommand = SENDOTPDELIVERYMETHOD_None; 
    my $isOtpAllowed= 0; 

    my $size = @$methods;

    SurePassID::WriteTrace($user, "ParseVerifyMethod() Entry"); 

    for($index=0; $index<$size; $index++)
	{  	
	    	WriteTrace($user, "GetVerifyMethods() methods[$index]=@$methods[$index]"); 

		if( @$methods[$index] eq 'OTP' ) 
		{
	    	   SurePassID::WriteTrace($user, "GetVerifyMethods() OTP yes "); 
		   $isOtpAllowed = 1; 
		} 
		elsif (@$methods[$index] eq 'PUSHVOICE') 
		{
	    	        SurePassID::WriteTrace($user, "GetVerifyMethods() PUSHVOICE yes "); 
			if ($defaultAuthCommand == SENDOTPDELIVERYMETHOD_None) 
			{
			 $defaultAuthCommand = SENDOTPDELIVERYMETHOD_VoicePushQuestion; 
			} 
		} 
                elsif (@$methods[$index] eq 'PUSHAPP') 
		{
	    	        WriteTrace($user, "GetVerifyMethods() PUSHAPP yes "); 
			if ($defaultAuthCommand == SENDOTPDELIVERYMETHOD_None) 
			{
			 $defaultAuthCommand = SENDOTPDELIVERYMETHOD_PushAppQuestion; 
			} 
		} 
	} 

     WriteTrace($user, "GetVerifyMethods() defaultAuthCommand = $defaultAuthCommand, isOtpAllowed=$isOtpAllowed "); 

     return ($defaultAuthCommand, $isOtpAllowed) 
} 

#############################################################################
# Unused subroutines called by FreeRadius. Must leave them here.
#############################################################################

# Function to handle preacct
sub preacct { return RLM_MODULE_NOOP; }
# Function to handle accounting
sub accounting { return RLM_MODULE_NOOP; }
# Function to handle checksimul
sub checksimul { return RLM_MODULE_NOOP; }
# Function to handle pre_proxy
sub pre_proxy { return RLM_MODULE_NOOP; }
# Function to handle post_proxy
sub post_proxy { return RLM_MODULE_NOOP; }
# Function to handle post_auth
sub post_auth { return RLM_MODULE_NOOP; }
# Function to handle xlat
sub xlat { return RLM_MODULE_NOOP }
#
sub authorize { return RLM_MODULE_NOOP }

